<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Cuti extends Model
{
    use HasFactory;

    protected $table = 'cuti';

    protected $fillable = [
        'karyawan_id',
        'jenis_id',
        'tanggal_mulai',
        'tanggal_selesai',
        'alasan',
        'file_pendukung',
        'status',
        'current_step',
        'is_bentrok',
        'catatan_admin'
    ];

    protected $casts = [
        'tanggal_mulai' => 'date',
        'tanggal_selesai' => 'date',
        'is_bentrok' => 'boolean'
    ];

    /**
     * Relasi ke Karyawan
     */
    public function karyawan()
    {
        return $this->belongsTo(Karyawan::class);
    }

    /**
     * Relasi ke Jenis Cuti
     */
    public function jenisCuti()
    {
        return $this->belongsTo(JenisCuti::class, 'jenis_id');
    }

    /**
     * Relasi ke Approval Steps
     */
    public function approvals()
    {
        return $this->hasMany(CutiApproval::class)->orderBy('id');
    }

    /**
     * Get current approval step
     */
    public function currentApproval()
    {
        return $this->hasOne(CutiApproval::class)
            ->where('step', $this->current_step)
            ->where('status', 'pending');
    }

    /**
     * Get jumlah hari cuti
     */
    public function getJumlahHariAttribute()
    {
        return $this->tanggal_mulai->diffInDays($this->tanggal_selesai) + 1;
    }

    /**
     * Check if user can approve current step
     */
    public function canBeApprovedBy($user)
    {
        if ($this->status !== 'pending') {
            return false;
        }

        // Cek apakah user adalah pemohon (self-approval tidak diizinkan)
        if ($this->karyawan->user_id === $user->id) {
            return false;
        }

        // Cek apakah role user sesuai dengan current step
        $roleStepMapping = [
            'manager' => 'manager',
            'gm' => 'gm',
            'hrd' => 'hrd',
            'super_admin' => 'hrd' // Super admin bisa approve sebagai HRD
        ];

        $requiredStep = $roleStepMapping[$user->role] ?? null;

        return $requiredStep && $requiredStep === $this->current_step;
    }

    /**
     * Get approval workflow steps based on requester role
     */
    public static function getApprovalSteps($requesterRole)
    {
        // Urutan approval berdasarkan role pengaju
        $workflows = [
            'karyawan' => ['manager', 'gm', 'hrd'],
            'manager' => ['gm', 'hrd'], // Manager tidak self-approve
            'gm' => ['hrd'], // GM tidak self-approve
            'hrd' => [], // HRD tidak perlu approval
            'super_admin' => [] // Super admin tidak perlu approval
        ];

        return $workflows[$requesterRole] ?? ['manager', 'gm', 'hrd'];
    }

    /**
     * Get progress percentage - FIXED!
     */
    public function getProgressPercentageAttribute()
    {
        if ($this->status === 'disetujui') {
            return 100;
        }

        if ($this->status === 'ditolak') {
            return 0;
        }

        $totalApprovals = $this->approvals()->count();
        if ($totalApprovals === 0) {
            return 0;
        }

        $completedApprovals = $this->approvals()->where('status', 'disetujui')->count();

        return round(($completedApprovals / $totalApprovals) * 100);
    }

    /**
     * Get status badge color
     */
    public function getStatusColorAttribute()
    {
        $colors = [
            'pending' => 'warning',
            'disetujui' => 'success',
            'ditolak' => 'danger'
        ];

        return $colors[$this->status] ?? 'secondary';
    }
}
